// ==UserScript==
// @name         LMArena API Bridge v3.2.6
// @namespace    http://tampermonkey.net/
// @version      3.2.6
// @description  Bridges LMArena to a local API server via WebSocket (with smart error handling & 429 protection)
// @author       Your Name
// @match        https://lmarena.ai/*
// @match        https://*.lmarena.ai/*
// @icon         https://www.google.com/s2/favicons?sz=64&domain=lmarena.ai
// @connect      localhost
// @connect      127.0.0.1
// @grant        none
// @run-at       document-end
// ==/UserScript==

(function () {
    'use strict';

    const SERVER_URL = "ws://localhost:5102/ws";
    let socket;
    let isCaptureModeActive = false;
    let is429Lockdown = false;

    // Session Timer Variables
    const SESSION_TIMEOUT_MS = 35 * 60 * 1000; // 35 minutes
    let sessionTimeoutId = null;

    // ============================================
    // Rate Limiting Configuration
    // ============================================

    const RATE_LIMIT = {
        maxRequests: 3,
        windowMs: 6.5 * 60 * 1000,
        enabled: true,
        countdownInterval: 30,
    };

    const COOLDOWN = {
        minIntervalMs: 60 * 1000,
        enabled: true,
        countdownInterval: 10,
    };

    let requestTimestamps = [];
    let lastRequestTime = 0;

    // ============================================
    // Queue Configuration
    // ============================================

    let queueEnabled = false;
    let QUEUE_DELAY_MS = 2 * 60 * 1000;

    let requestQueue = [];
    let isProcessingQueue = false;
    let queueCountdown = 0;

    // ============================================
    // Countdown tracking
    // ============================================
    let activeCountdowns = {
        cooldown: null,
        rateLimit: null,
    };

    // ============================================
    // Captured reCAPTCHA parameters
    // ============================================
    let capturedSiteKey = null;
    let capturedAction = null;

    function uuidv7() {
        const timestamp = Date.now();
        const timestampHex = timestamp.toString(16).padStart(12, '0');
        const rand_a = Math.floor(Math.random() * 4096);
        const rand_b_1 = Math.floor(Math.random() * 16384);
        const rand_b_2 = Math.floor(Math.random() * 65536);
        const rand_b_3 = Math.floor(Math.random() * 65536);
        const rand_b_4 = Math.floor(Math.random() * 65536);

        return (
            timestampHex.substring(0, 8) + '-' +
            timestampHex.substring(8, 12) + '-' +
            '7' + rand_a.toString(16).padStart(3, '0') + '-' +
            ((rand_b_1 & 0x3fff) | 0x8000).toString(16).padStart(4, '0') + '-' +
            (rand_b_2.toString(16).padStart(4, '0') +
             rand_b_3.toString(16).padStart(4, '0') +
             rand_b_4.toString(16).padStart(4, '0'))
        );
    }

    function sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    // ============================================
    // Helper: System Status Check
    // ============================================
    function isSystemBusy() {
        if (is429Lockdown) return "429 Lockdown Mode";
        if (activeCountdowns.cooldown !== null) return "Active Cooldown";
        if (activeCountdowns.rateLimit !== null) return "Active Rate Limit Window";
        if (isProcessingQueue) return "Queue Processing";
        if (requestQueue.length > 0) return "Queue Not Empty";
        return false;
    }

    // ============================================
    // Session Timer Logic
    // ============================================
    function resetSessionTimer() {
        if (sessionTimeoutId) {
            clearTimeout(sessionTimeoutId);
        }

        sessionTimeoutId = setTimeout(() => {
            const warningMsg = "⚠️ Session idle for 35m. LMArena connection likely stale (TypeErrors imminent). Refresh page or send manual message.";
            console.warn(`[API Bridge] ${warningMsg}`);
            if (socket && socket.readyState === WebSocket.OPEN) {
                socket.send(JSON.stringify({
                    request_id: "system_warning",
                    data: { warning: warningMsg }
                }));
            }
        }, SESSION_TIMEOUT_MS);
    }

    // ============================================
    // Countdown Functions
    // ============================================

    function startCooldownCountdown() {
        if (activeCountdowns.cooldown) {
            clearInterval(activeCountdowns.cooldown);
        }

        const endTime = lastRequestTime + COOLDOWN.minIntervalMs;

        activeCountdowns.cooldown = setInterval(() => {
            const remaining = Math.ceil((endTime - Date.now()) / 1000);

            if (remaining <= 0) {
                console.log("[API Bridge] ⏱️ Cooldown: Ready");
                clearInterval(activeCountdowns.cooldown);
                activeCountdowns.cooldown = null;
            } else if (remaining % COOLDOWN.countdownInterval === 0) {
                console.log(`[API Bridge] ⏱️ Cooldown: ${remaining}s remaining`);
            }
        }, 1000);
    }

    function startRateLimitCountdown() {
        if (activeCountdowns.rateLimit) {
            clearInterval(activeCountdowns.rateLimit);
        }

        activeCountdowns.rateLimit = setInterval(() => {
            cleanupOldTimestamps();

            if (requestTimestamps.length < RATE_LIMIT.maxRequests) {
                console.log(`[API Bridge] 📊 Rate limit: Ready (${requestTimestamps.length}/${RATE_LIMIT.maxRequests} used)`);
                clearInterval(activeCountdowns.rateLimit);
                activeCountdowns.rateLimit = null;
            } else {
                const oldestTimestamp = Math.min(...requestTimestamps);
                const remaining = Math.ceil((oldestTimestamp + RATE_LIMIT.windowMs - Date.now()) / 1000);

                if (remaining > 0 && remaining % RATE_LIMIT.countdownInterval === 0) {
                    console.log(`[API Bridge] 📊 Rate limit: ${remaining}s until slot available (${requestTimestamps.length}/${RATE_LIMIT.maxRequests} used)`);
                }
            }
        }, 1000);
    }

    function stopAllCountdowns() {
        if (activeCountdowns.cooldown) {
            clearInterval(activeCountdowns.cooldown);
            activeCountdowns.cooldown = null;
        }
        if (activeCountdowns.rateLimit) {
            clearInterval(activeCountdowns.rateLimit);
            activeCountdowns.rateLimit = null;
        }
    }

    // ============================================
    // Window-based Rate Limiting Functions
    // ============================================

    function cleanupOldTimestamps() {
        const now = Date.now();
        const cutoff = now - RATE_LIMIT.windowMs;
        requestTimestamps = requestTimestamps.filter(ts => ts > cutoff);
    }

    function canMakeRequestWindow() {
        if (!RATE_LIMIT.enabled) return true;
        cleanupOldTimestamps();
        return requestTimestamps.length < RATE_LIMIT.maxRequests;
    }

    function getTimeUntilNextSlot() {
        if (!RATE_LIMIT.enabled) return 0;
        cleanupOldTimestamps();

        if (requestTimestamps.length < RATE_LIMIT.maxRequests) {
            return 0;
        }

        const oldestTimestamp = Math.min(...requestTimestamps);
        const timeUntilFree = (oldestTimestamp + RATE_LIMIT.windowMs) - Date.now();

        return Math.max(0, timeUntilFree);
    }

    function getWindowRateLimitStatus() {
        cleanupOldTimestamps();
        const remaining = RATE_LIMIT.maxRequests - requestTimestamps.length;
        const waitTime = getTimeUntilNextSlot();

        return {
            enabled: RATE_LIMIT.enabled,
            used: requestTimestamps.length,
            max: RATE_LIMIT.maxRequests,
            remaining: remaining,
            waitTimeMs: waitTime,
            waitTimeSec: Math.ceil(waitTime / 1000),
        };
    }

    // ============================================
    // Cooldown Rate Limiting Functions
    // ============================================

    function canMakeRequestCooldown() {
        if (!COOLDOWN.enabled) return true;
        const now = Date.now();
        const timeSinceLastRequest = now - lastRequestTime;
        return timeSinceLastRequest >= COOLDOWN.minIntervalMs;
    }

    function getCooldownRemaining() {
        if (!COOLDOWN.enabled) return 0;
        const now = Date.now();
        const timeSinceLastRequest = now - lastRequestTime;
        const remaining = COOLDOWN.minIntervalMs - timeSinceLastRequest;
        return Math.max(0, remaining);
    }

    function getCooldownStatus() {
        const remainingMs = getCooldownRemaining();
        return {
            enabled: COOLDOWN.enabled,
            ready: remainingMs === 0,
            remainingMs: remainingMs,
            remainingSec: Math.ceil(remainingMs / 1000),
            intervalMs: COOLDOWN.minIntervalMs,
            intervalSec: COOLDOWN.minIntervalMs / 1000,
        };
    }

    // ============================================
    // Combined Rate Limit Check
    // ============================================

    function checkRateLimits() {
        if (!canMakeRequestCooldown()) {
            const cooldownStatus = getCooldownStatus();
            return {
                allowed: false,
                reason: 'cooldown',
                waitTimeSec: cooldownStatus.remainingSec,
                waitTimeMs: cooldownStatus.remainingMs,
            };
        }

        if (!canMakeRequestWindow()) {
            const windowStatus = getWindowRateLimitStatus();
            return {
                allowed: false,
                reason: 'window',
                waitTimeSec: windowStatus.waitTimeSec,
                waitTimeMs: windowStatus.waitTimeMs,
                windowStatus: windowStatus,
            };
        }

        return { allowed: true };
    }

    function recordRequest() {
        const now = Date.now();

        if (COOLDOWN.enabled) {
            lastRequestTime = now;
            startCooldownCountdown();
        }

        if (RATE_LIMIT.enabled) {
            cleanupOldTimestamps();
            requestTimestamps.push(now);
            console.log(`[API Bridge] 📊 Rate limit: ${requestTimestamps.length}/${RATE_LIMIT.maxRequests} in window`);
            if (requestTimestamps.length >= RATE_LIMIT.maxRequests) {
                startRateLimitCountdown();
            }
        }
    }

    function rollbackRateLimit() {
        if (RATE_LIMIT.enabled && requestTimestamps.length > 0) {
            requestTimestamps.pop();
            console.log("[API Bridge] ↩️ Error detected: Rate limit count rolled back (Window limit restored, Cooldown remains)");
        }
    }

    // ============================================
    // 429 LOCKDOWN / AUTO-PROTECTION
    // ============================================
    function triggerAutoProtection() {
        if (is429Lockdown) return;

        is429Lockdown = true;
        console.warn("[API Bridge] 🚨 429 DETECTED! Engaging Lockdown Protection Protocols!");
        console.warn("[API Bridge] 🛡️ Queue, Window Limit, and Cooldown are now LOCKED to ON.");

        // Add 2 minute penalty to the window
        RATE_LIMIT.windowMs += (2 * 60 * 1000);
        console.warn(`[API Bridge] 📉 Penalty: Rate limit window increased by 2 minutes (New: ${RATE_LIMIT.windowMs / 1000}s).`);

        // Lock UI settings
        window.enableAllLimits();

        // 1. Force fill the rate limit bucket
        const now = Date.now();
        lastRequestTime = now;
        requestTimestamps = new Array(RATE_LIMIT.maxRequests).fill(now);

        startCooldownCountdown();
        startRateLimitCountdown();

        // 2. Engage Anti-Refresh Shield (BeforeUnload + Overlay)
        engageLockdownShield();
    }

    function engageLockdownShield() {
        // Prevent accidental closes/refreshes
        window.addEventListener('beforeunload', function (e) {
            if (is429Lockdown) {
                e.preventDefault();
                e.returnValue = 'System is in 429 Lockdown. Fuck you for being an ungrateful locust.';
                return e.returnValue;
            }
        });

        // Visual warning overlay
        const overlay = document.createElement('div');
        overlay.id = 'lmarena-lockdown-overlay';
        overlay.style.cssText = `
            position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(20, 0, 0, 0.9); color: #ff4444; z-index: 2147483647;
            display: flex; flex-direction: column; align-items: center; justify-content: center;
            font-family: monospace; text-align: center; pointer-events: none;
            backdrop-filter: blur(5px);
        `;
        overlay.innerHTML = `
            <h1 style="font-size: 48px; margin-bottom: 20px;">🚫 429 LOCKDOWN ACTIVE 🚫</h1>
            <h2 style="color: white; margin-bottom: 40px;">DO NOT REFRESH THE PAGE YOU FILTHY LOCUST~</h2>
            <div style="background: rgba(0,0,0,0.5); padding: 20px; border: 1px solid #ff4444; border-radius: 8px;">
                <p style="font-size: 18px; color: #ffcccc; margin: 5px;">Total Locust Death</p>
                <p style="font-size: 18px; color: #ffcccc; margin: 5px;">Kill locusts. Behead locusts. Roundhouse kick a locust into the concrete. Slam dunk a locust baby into the trashcan. Crucify filthy locusts. Defecate in a locusts food. Launch locusts into the sun. Stir fry locusts in a wok. Toss locusts into active volcanoes. Urinate into a locusts gas tank. Judo throw locusts into a wood chipper. Twist locusts heads off. Report locusts to the IRS. Karate chop locusts in half. Curb stomp pregnant black locusts. Trap locusts in quicksand. Crush locusts in the trash compactor. Liquefy locusts in a vat of acid. Eat locusts. Dissect locusts. Exterminate locusts in the gas chamber. Stomp locust skulls with steel toed boots. Cremate locusts in the oven. Lobotomize locusts. Mandatory abortions for locusts. Grind locust fetuses in the garbage disposal. Drown locusts in fried chicken grease. Vaporize locusts with a ray gun. Kick old locusts down the stairs. Feed locusts to alligators. Slice locusts with a katana.</p>
            </div>
        `;
        document.body.appendChild(overlay);
    }

    // ============================================
    // Queue Functions
    // ============================================

    function getEstimatedWaitTime(position) {
        const baseWait = position * (QUEUE_DELAY_MS / 1000);
        return baseWait + queueCountdown;
    }

    function addToQueue(requestId, payload) {
        const queueItem = {
            requestId: requestId,
            payload: payload,
            addedAt: Date.now(),
        };

        requestQueue.push(queueItem);

        const position = requestQueue.length;
        const estimatedWaitSec = getEstimatedWaitTime(position);

        console.log(`[API Bridge] 📥 Added to queue. Position: ${position}, Est. wait: ${estimatedWaitSec}s`);

        sendToServer(requestId, `QUEUED:${estimatedWaitSec}:${position}`);

        if (!isProcessingQueue) {
            startQueueProcessor();
        }
    }

    async function startQueueProcessor() {
        if (isProcessingQueue) {
            return;
        }

        isProcessingQueue = true;
        console.log("[API Bridge] 🚀 Queue processor started");

        while (requestQueue.length > 0) {
            // STEP 1: Check rate limits FIRST
            const rateLimitCheck = checkRateLimits();

            if (!rateLimitCheck.allowed) {
                const failedItem = requestQueue.shift();
                console.warn(`[API Bridge] ⚠️ Queue blocked by ${rateLimitCheck.reason}. Rejecting request ${failedItem.requestId.substring(0,8)}.`);

                if (rateLimitCheck.reason === 'cooldown') {
                    sendToServer(failedItem.requestId, { error: `Slow Down Locust! Wait ${rateLimitCheck.waitTimeSec}s` });
                } else {
                    sendToServer(failedItem.requestId, { error: `You have exceeded the rate limit. Try not swiping too much next time. Wait ${rateLimitCheck.waitTimeSec}s` });
                }
                sendToServer(failedItem.requestId, "[DONE]");

                continue;
            }

            // STEP 2: Wait the queue delay
            if (QUEUE_DELAY_MS > 0) {
                console.log(`[API Bridge] ⏳ Queue: Waiting ${QUEUE_DELAY_MS / 1000}s...`);
                queueCountdown = Math.ceil(QUEUE_DELAY_MS / 1000);

                while (queueCountdown > 0) {
                    await sleep(1000);
                    queueCountdown--;

                    if (queueCountdown > 0 && queueCountdown % 30 === 0) {
                        console.log(`[API Bridge] ⏳ Queue: ${queueCountdown}s remaining...`);
                        requestQueue.forEach((item, index) => {
                            const estimatedWaitSec = getEstimatedWaitTime(index + 1);
                            sendToServer(item.requestId, `QUEUED:${estimatedWaitSec}:${index + 1}`);
                        });
                    }
                }
            }

            if (requestQueue.length === 0) {
                break;
            }

            // STEP 3: Process
            const item = requestQueue.shift();
            if (!item) break;

            console.log(`[API Bridge] 🚀 Processing queued request ${item.requestId.substring(0, 8)} (${requestQueue.length} remaining)`);
            sendToServer(item.requestId, `PROCESSING:0`);

            await handleRequest(item.requestId, item.payload);
        }

        isProcessingQueue = false;
        queueCountdown = 0;
        console.log("[API Bridge] ✅ Queue processor finished");
    }

    // ============================================
    // Console Commands
    // ============================================

    window.help = function() {
        const queueState = queueEnabled ? 'ON ' : 'OFF';
        const rateLimitState = RATE_LIMIT.enabled ? 'ON ' : 'OFF';
        const cooldownState = COOLDOWN.enabled ? 'ON ' : 'OFF';
        const lockdownStatus = is429Lockdown ? '🔒 LOCKED (429 HIT)' : '🔓 Unlocked';
        const busyStatus = isSystemBusy() ? `⚠️ BUSY: ${isSystemBusy()}` : '✅ IDLE';

        console.log("╔══════════════════════════════════════════════════════════════╗");
        console.log("║           LMArena API Bridge v3.2.6 - Help                   ║");
        console.log("╠══════════════════════════════════════════════════════════════╣");
        console.log("║                                                              ║");
        console.log(`║  STATUS: ${lockdownStatus} | ${busyStatus}${' '.repeat(Math.max(0, 35 - lockdownStatus.length - busyStatus.length))}║`);
        console.log("║                                                              ║");
        console.log("║  CURRENT SETTINGS:                                           ║");
        console.log(`║    📊 Rate Limit: ${RATE_LIMIT.maxRequests} req / ${(RATE_LIMIT.windowMs / 1000 / 60).toFixed(1)} min [${rateLimitState}]                    ║`);
        console.log(`║    ⏱️  Cooldown:   ${COOLDOWN.minIntervalMs / 1000}s between requests [${cooldownState}]                ║`);
        console.log(`║    📥 Queue:      ${queueState} (${QUEUE_DELAY_MS / 1000}s delay)                            ║`);
        console.log("║                                                              ║");
        console.log("╠══════════════════════════════════════════════════════════════╣");
        console.log("║  ADJUSTMENT COMMANDS (Locked when Busy/429):                 ║");
        console.log("║    setQueueDelay(sec)  - Set delay between queued items      ║");
        console.log("║    setCooldown(sec)    - Set cooldown between requests       ║");
        console.log("║    setRateLimit(req, sec) - Set window limit (e.g. 3, 200)   ║");
        console.log("║                                                              ║");
        console.log("╠══════════════════════════════════════════════════════════════╣");
        console.log("║  QUEUE COMMANDS:                                             ║");
        console.log("║    enableQueue()     - Turn queue ON                         ║");
        console.log("║    disableQueue()    - Turn queue OFF                        ║");
        console.log("║    toggleQueue()     - Toggle queue state                    ║");
        console.log("║    queueStatus()     - View queue details                    ║");
        console.log("║    clearQueue()      - Clear all queued requests             ║");
        console.log("║                                                              ║");
        console.log("╠══════════════════════════════════════════════════════════════╣");
        console.log("║  RATE LIMIT COMMANDS:                                        ║");
        console.log("║    enableRateLimit()  - Turn window rate limit ON            ║");
        console.log("║    disableRateLimit() - Turn window rate limit OFF           ║");
        console.log("║    enableCooldown()   - Turn cooldown ON                     ║");
        console.log("║    disableCooldown()  - Turn cooldown OFF                    ║");
        console.log("║    disableAllLimits() - Disable ALL (incl. queue) ⚠️         ║");
        console.log("║    enableAllLimits()  - Enable ALL (incl. queue)             ║");
        console.log("║                                                              ║");
        console.log("╠══════════════════════════════════════════════════════════════╣");
        console.log("║  STATUS COMMANDS:                                            ║");
        console.log("║    rateLimitStatus()  - View rate limit status               ║");
        console.log("║    debugRecaptcha()   - Debug reCAPTCHA state                ║");
        console.log("║                                                              ║");
        console.log("╚══════════════════════════════════════════════════════════════╝");

        return "Type any command above to execute it.";
    };

    // Adjustment Commands
    window.setQueueDelay = function(seconds) {
        const busyReason = isSystemBusy();
        if (busyReason) { console.error(`⛔ Cannot adjust settings: System is ${busyReason}. Wait for idle.`); return; }
        if (!seconds || seconds <= 0) { console.error("❌ Value cannot be 0 or negative. Use disableQueue() instead."); return; }
        QUEUE_DELAY_MS = seconds * 1000;
        console.log(`[API Bridge] ✅ Queue delay set to ${seconds}s`);
    };

    window.setCooldown = function(seconds) {
        const busyReason = isSystemBusy();
        if (busyReason) { console.error(`⛔ Cannot adjust settings: System is ${busyReason}. Wait for idle.`); return; }
        if (!seconds || seconds <= 0) { console.error("❌ Value cannot be 0 or negative. Use disableCooldown() instead."); return; }
        COOLDOWN.minIntervalMs = seconds * 1000;
        console.log(`[API Bridge] ✅ Cooldown set to ${seconds}s`);
    };

    window.setRateLimit = function(requests, seconds) {
        const busyReason = isSystemBusy();
        if (busyReason) { console.error(`⛔ Cannot adjust settings: System is ${busyReason}. Wait for idle.`); return; }
        if (!requests || requests <= 0 || !seconds || seconds <= 0) { console.error("❌ Values cannot be 0 or negative. Use disableRateLimit() instead."); return; }
        RATE_LIMIT.maxRequests = requests;
        RATE_LIMIT.windowMs = seconds * 1000;
        console.log(`[API Bridge] ✅ Rate limit set to ${requests} requests per ${seconds}s`);
    };

    // Queue commands
    window.enableQueue = function() {
        queueEnabled = true;
        console.log("[API Bridge] ✅ Queue ENABLED");
        console.log(`[API Bridge]    All requests will be queued with ${QUEUE_DELAY_MS / 1000}s delay`);
        return { enabled: true };
    };

    window.disableQueue = function() {
        if (is429Lockdown) { console.error("⛔ Action blocked: System is in 429 Lockdown. Try not being an ungrateful locust next time."); return; }
        queueEnabled = false;
        console.log("[API Bridge] ❌ Queue DISABLED");
        return { enabled: false };
    };

    window.toggleQueue = function() {
        if (queueEnabled) {
            return window.disableQueue();
        } else {
            return window.enableQueue();
        }
    };

    window.queueStatus = function() {
        console.log("=== Queue Status ===");
        console.log(`Enabled: ${queueEnabled}`);
        console.log(`Queue length: ${requestQueue.length}`);
        console.log(`Processing: ${isProcessingQueue}`);
        console.log(`Delay between requests: ${QUEUE_DELAY_MS / 1000}s`);
        if (queueCountdown > 0) {
            console.log(`Next request in: ${queueCountdown}s`);
        }

        if (requestQueue.length > 0) {
            console.log("\nQueued requests:");
            requestQueue.forEach((item, index) => {
                const waitTime = getEstimatedWaitTime(index + 1);
                console.log(`  ${index + 1}. ${item.requestId.substring(0, 8)}... (est. ${waitTime}s)`);
            });
        }

        return {
            enabled: queueEnabled,
            length: requestQueue.length,
            processing: isProcessingQueue,
            delayMs: QUEUE_DELAY_MS,
            countdown: queueCountdown,
        };
    };

    window.clearQueue = function() {
        const count = requestQueue.length;

        requestQueue.forEach(item => {
            sendToServer(item.requestId, { error: "QUEUE_CLEARED" });
            sendToServer(item.requestId, "[DONE]");
        });

        requestQueue = [];
        console.log(`[API Bridge] 🗑️ Cleared ${count} items from queue`);

        return { cleared: count };
    };

    // Rate limit commands
    window.enableRateLimit = function() {
        RATE_LIMIT.enabled = true;
        console.log("[API Bridge] ✅ Window rate limit ENABLED");
        return { enabled: true };
    };

    window.disableRateLimit = function() {
        if (is429Lockdown) { console.error("⛔ Action blocked: System is in 429 Lockdown. Try not being an ungrateful locust next time."); return; }
        RATE_LIMIT.enabled = false;
        if (activeCountdowns.rateLimit) {
            clearInterval(activeCountdowns.rateLimit);
            activeCountdowns.rateLimit = null;
        }
        console.log("[API Bridge] ⚠️  Window rate limit DISABLED");
        return { enabled: false };
    };

    window.enableCooldown = function() {
        COOLDOWN.enabled = true;
        console.log("[API Bridge] ✅ Cooldown ENABLED");
        return { enabled: true };
    };

    window.disableCooldown = function() {
        if (is429Lockdown) { console.error("⛔ Action blocked: System is in 429 Lockdown. Try not being an ungrateful locust next time."); return; }
        COOLDOWN.enabled = false;
        if (activeCountdowns.cooldown) {
            clearInterval(activeCountdowns.cooldown);
            activeCountdowns.cooldown = null;
        }
        console.log("[API Bridge] ⚠️  Cooldown DISABLED");
        return { enabled: false };
    };

    window.disableAllLimits = function() {
        if (is429Lockdown) { console.error("⛔ Action blocked: System is in 429 Lockdown. Try not being an ungrateful locust next time."); return; }
        RATE_LIMIT.enabled = false;
        COOLDOWN.enabled = false;
        queueEnabled = false;
        stopAllCountdowns();

        console.log("[API Bridge] ⚠️⚠️⚠️ ALL LIMITS DISABLED ⚠️⚠️⚠️");
        return { rateLimitEnabled: false, cooldownEnabled: false, queueEnabled: false };
    };

    window.enableAllLimits = function() {
        RATE_LIMIT.enabled = true;
        COOLDOWN.enabled = true;
        queueEnabled = true;

        console.log("[API Bridge] ✅ ALL LIMITS ENABLED");
        return { rateLimitEnabled: true, cooldownEnabled: true, queueEnabled: true };
    };

    window.rateLimitStatus = function() {
        const windowStatus = getWindowRateLimitStatus();
        const cooldownStatus = getCooldownStatus();

        console.log("=== Rate Limit Status ===");
        if (is429Lockdown) console.log("🔒 SYSTEM LOCKED DOWN (429 Hit)");
        console.log("");
        console.log("Window limit:");
        console.log(`  Enabled: ${RATE_LIMIT.enabled ? '✅ ON' : '❌ OFF'}`);
        if (RATE_LIMIT.enabled) {
            console.log(`  Requests in window: ${windowStatus.used}/${windowStatus.max}`);
            console.log(`  Remaining: ${windowStatus.remaining}`);
            if (windowStatus.waitTimeMs > 0) {
                console.log(`  Next slot in: ${windowStatus.waitTimeSec}s`);
            } else {
                console.log(`  Window: Ready`);
            }
            console.log(`  Window duration: ${RATE_LIMIT.windowMs / 1000}s`);
        }
        console.log("");
        console.log("Cooldown limit:");
        console.log(`  Enabled: ${COOLDOWN.enabled ? '✅ ON' : '❌ OFF'}`);
        if (COOLDOWN.enabled) {
            console.log(`  Min interval: ${COOLDOWN.minIntervalMs / 1000}s`);
            if (cooldownStatus.ready) {
                console.log(`  Cooldown: Ready`);
            } else {
                console.log(`  Cooldown remaining: ${cooldownStatus.remainingSec}s`);
            }
        }
        console.log("");
        console.log("Queue:");
        console.log(`  Enabled: ${queueEnabled ? '✅ ON' : '❌ OFF'}`);
        console.log(`  Items in queue: ${requestQueue.length}`);
        if (queueCountdown > 0) {
            console.log(`  Next processing in: ${queueCountdown}s`);
        }
        console.log("");

        const check = checkRateLimits();
        console.log(`Overall: ${check.allowed ? '✅ Ready' : '❌ Wait ' + check.waitTimeSec + 's (' + check.reason + ')'}`);

        return { windowStatus, cooldownStatus, check, queue: { enabled: queueEnabled, length: requestQueue.length } };
    };

    // ============================================
    // Hook grecaptcha.execute
    // ============================================
    function hookGrecaptcha() {
        const tryHook = () => {
            const captcha = window.grecaptcha?.enterprise || window.grecaptcha;

            if (!captcha || !captcha.execute) {
                return false;
            }

            if (captcha._hooked) {
                return true;
            }

            const originalExecute = captcha.execute.bind(captcha);

            captcha.execute = function(siteKey, options) {
                console.log("[API Bridge] 🔍 Intercepted grecaptcha.execute:");
                console.log("[API Bridge]    Site Key:", siteKey);
                console.log("[API Bridge]    Options:", JSON.stringify(options));

                if (siteKey && typeof siteKey === 'string' && siteKey.length > 10) {
                    capturedSiteKey = siteKey;
                    if (options && options.action) {
                        capturedAction = options.action;
                    }
                    console.log("[API Bridge] ✅ Captured reCAPTCHA params - Key:", siteKey.substring(0, 10) + "..., Action:", capturedAction);
                }

                return originalExecute(siteKey, options);
            };

            captcha._hooked = true;
            console.log("[API Bridge] ✅ Successfully hooked grecaptcha.execute");
            return true;
        };

        if (tryHook()) return;

        let attempts = 0;
        const interval = setInterval(() => {
            attempts++;
            if (tryHook() || attempts > 30) {
                clearInterval(interval);
                if (attempts > 30) {
                    console.warn("[API Bridge] ⚠️ Could not hook grecaptcha after 30 attempts");
                }
            }
        }, 1000);
    }

    hookGrecaptcha();

    // ============================================
    // Find reCAPTCHA site key
    // ============================================
    function findRecaptchaSiteKey() {
        if (capturedSiteKey) {
            console.log("[API Bridge] Using captured site key");
            return capturedSiteKey;
        }

        if (window.___grecaptcha_cfg && window.___grecaptcha_cfg.clients) {
            console.log("[API Bridge] Checking ___grecaptcha_cfg.clients...");
            for (const [clientId, client] of Object.entries(window.___grecaptcha_cfg.clients)) {
                console.log("[API Bridge]   Client", clientId, ":", typeof client);
                if (client && typeof client === 'object') {
                    const findKey = (obj, path = '', depth = 0) => {
                        if (!obj || typeof obj !== 'object' || depth > 15) return null;

                        for (const keyName of ['sitekey', 'siteKey', 'key', 'k']) {
                            if (obj[keyName] && typeof obj[keyName] === 'string' && obj[keyName].length > 20) {
                                console.log(`[API Bridge]   Found key at ${path}.${keyName}`);
                                return obj[keyName];
                            }
                        }

                        for (const [k, val] of Object.entries(obj)) {
                            if (typeof val === 'object') {
                                const found = findKey(val, `${path}.${k}`, depth + 1);
                                if (found) return found;
                            }
                        }
                        return null;
                    };
                    const key = findKey(client, `clients.${clientId}`);
                    if (key) return key;
                }
            }
        }

        const scripts = document.querySelectorAll('script[src*="recaptcha"]');
        console.log("[API Bridge] Found", scripts.length, "recaptcha scripts");
        for (const script of scripts) {
            console.log("[API Bridge]   Script src:", script.src);
            const match = script.src.match(/render=([A-Za-z0-9_-]+)/);
            if (match && match[1] !== 'explicit' && match[1].length > 20) {
                console.log("[API Bridge]   Found key in script:", match[1]);
                return match[1];
            }
        }

        const element = document.querySelector('[data-sitekey]');
        if (element) {
            console.log("[API Bridge] Found data-sitekey:", element.getAttribute('data-sitekey'));
            return element.getAttribute('data-sitekey');
        }

        console.warn("[API Bridge] ❌ Could not find reCAPTCHA site key");
        return null;
    }

    // ============================================
    // Get fresh reCAPTCHA token
    // ============================================
    async function getFreshRecaptchaToken() {
        return new Promise((resolve) => {
            const captcha = window.grecaptcha?.enterprise || window.grecaptcha;

            console.log("[API Bridge] grecaptcha available:", !!captcha);
            console.log("[API Bridge] grecaptcha.enterprise:", !!window.grecaptcha?.enterprise);

            if (typeof captcha === 'undefined' || !captcha.execute) {
                console.warn("[API Bridge] grecaptcha not available, using cached token");
                resolve(window.recaptchaToken || "");
                return;
            }

            const siteKey = findRecaptchaSiteKey();
            const action = capturedAction || 'submit';

            console.log("[API Bridge] Using site key:", siteKey ? siteKey.substring(0, 15) + "..." : "null");
            console.log("[API Bridge] Using action:", action);

            if (!siteKey) {
                console.warn("[API Bridge] No site key, using cached token");
                resolve(window.recaptchaToken || "");
                return;
            }

            const timeoutId = setTimeout(() => {
                console.warn("[API Bridge] reCAPTCHA timeout after 10s");
                resolve(window.recaptchaToken || "");
            }, 10000);

            try {
                captcha.ready(function() {
                    console.log("[API Bridge] grecaptcha is ready, executing...");
                    captcha.execute(siteKey, { action: action })
                        .then(function(token) {
                            clearTimeout(timeoutId);
                            console.log("[API Bridge] ✅ Got fresh token:", token.substring(0, 20) + "...");
                            window.recaptchaToken = token;
                            resolve(token);
                        })
                        .catch(function(error) {
                            clearTimeout(timeoutId);
                            console.error("[API Bridge] ❌ grecaptcha.execute error:", error);
                            resolve(window.recaptchaToken || "");
                        });
                });
            } catch (error) {
                clearTimeout(timeoutId);
                console.error("[API Bridge] ❌ grecaptcha.ready error:", error);
                resolve(window.recaptchaToken || "");
            }
        });
    }

    window.debugRecaptcha = async function() {
        console.log("=== reCAPTCHA Debug Info ===");
        console.log("grecaptcha:", typeof window.grecaptcha);
        console.log("grecaptcha.enterprise:", typeof window.grecaptcha?.enterprise);
        console.log("___grecaptcha_cfg:", window.___grecaptcha_cfg);
        console.log("Captured site key:", capturedSiteKey);
        console.log("Captured action:", capturedAction);
        console.log("Cached token:", window.recaptchaToken ? window.recaptchaToken.substring(0, 30) + "..." : "none");

        const foundKey = findRecaptchaSiteKey();
        console.log("Found site key:", foundKey);

        console.log("\nAttempting to get fresh token...");
        const token = await getFreshRecaptchaToken();
        console.log("Result:", token ? token.substring(0, 30) + "..." : "FAILED");

        return { capturedSiteKey, capturedAction, foundKey, token };
    };

    // ============================================
    // WebSocket Connection
    // ============================================
    function connect() {
        console.log(`[API Bridge] Connecting to ${SERVER_URL}...`);
        socket = new WebSocket(SERVER_URL);

        socket.onopen = () => {
            console.log("[API Bridge] ✅ WebSocket connection established.");
            document.title = "✅ " + document.title;
            resetSessionTimer();
        };

        socket.onmessage = async (event) => {
            try {
                const message = JSON.parse(event.data);

                if (message.command) {
                    console.log(`[API Bridge] ⬇️ Received command: ${message.command}`);

                    if (message.command === 'refresh' || message.command === 'reconnect') {
                        if (is429Lockdown) {
                            console.warn("[API Bridge] ⛔ Refresh blocked: System is in 429 Lockdown. Refresh page manually to reset if needed.");
                            return;
                        }
                        console.log(`[API Bridge] Received '${message.command}' command, refreshing page...`);
                        location.reload();
                    } else if (message.command === 'activate_id_capture') {
                        console.log("[API Bridge] ✅ ID capture mode activated. Please trigger a 'Retry' action on the page.");
                        isCaptureModeActive = true;
                        document.title = "🎯 " + document.title;
                    } else if (message.command === 'send_page_source') {
                        console.log("[API Bridge] Received send_page_source command, sending...");
                        sendPageSource();
                    } else if (message.command === 'enable_queue') {
                        window.enableQueue();
                    } else if (message.command === 'disable_queue') {
                        window.disableQueue();
                    } else if (message.command === 'queue_status') {
                        const status = window.queueStatus();
                        sendToServer("system", JSON.stringify({ queue_status: status }));
                    } else if (message.command === 'clear_queue') {
                        window.clearQueue();
                    }
                    return;
                }

                const { request_id, payload } = message;

                if (!request_id || !payload) {
                    console.error("[API Bridge] Received invalid message from server:", message);
                    return;
                }

                console.log(`[API Bridge] ⬇️ Received chat request ${request_id.substring(0, 8)}.`);

                if (queueEnabled) {
                    console.log(`[API Bridge] 📥 Queue is ON - adding request to queue`);
                    addToQueue(request_id, payload);
                    return;
                }

                const rateLimitCheck = checkRateLimits();

                if (!rateLimitCheck.allowed) {
                    if (rateLimitCheck.reason === 'cooldown') {
                        console.warn(`[API Bridge] ⚠️ Cooldown active. Wait ${rateLimitCheck.waitTimeSec}s`);

                        sendToServer(request_id, { error: `Slow Down Locust! Wait: ${rateLimitCheck.waitTimeSec}s` });
                        sendToServer(request_id, "[DONE]");
                        return;

                    } else if (rateLimitCheck.reason === 'window') {
                        console.warn(`[API Bridge] ⚠️ Window rate limit hit. Wait ${rateLimitCheck.waitTimeSec}s`);

                        sendToServer(request_id, { error: `You have exceeded the rate limit. Try not swiping too much next time. Wait: ${rateLimitCheck.waitTimeSec}s` });
                        sendToServer(request_id, "[DONE]");
                        return;
                    }
                }

                await handleRequest(request_id, payload);

            } catch (error) {
                console.error("[API Bridge] Error processing server message:", error);
            }
        };

        socket.onclose = () => {
            console.warn("[API Bridge] 🔌 Connection closed. Reconnecting in 5s...");
            if (document.title.startsWith("✅ ")) {
                document.title = document.title.substring(2);
            }
            setTimeout(connect, 5000);
        };

        socket.onerror = (error) => {
            console.error("[API Bridge] ❌ WebSocket error:", error);
            socket.close();
        };
    }

    // ============================================
    // Request Handler
    // ============================================
    async function handleRequest(requestId, payload) {
        resetSessionTimer(); // Reset inactivity timer on new request

        const { message_templates, target_model_id, session_id, message_id } = payload;

        if (!session_id || !message_id) {
            const errorMsg = "Session IDs missing. Please run setup again.";
            console.error(`[API Bridge] ${errorMsg}`);
            sendToServer(requestId, { error: errorMsg });
            sendToServer(requestId, "[DONE]");
            return;
        }

        if (!message_templates || message_templates.length === 0) {
            const errorMsg = "No messages to send.";
            console.error(`[API Bridge] ${errorMsg}`);
            sendToServer(requestId, { error: errorMsg });
            sendToServer(requestId, "[DONE]");
            return;
        }

        const apiUrl = `/nextjs-api/stream/create-evaluation`;
        const evaluationId = uuidv7();
        await new Promise(resolve => setTimeout(resolve, 45));

        const newMessages = [];
        let lastMsgId = null;

        for (let i = 0; i < message_templates.length; i++) {
            const template = message_templates[i];
            const currentMsgId = uuidv7();
            const parentIds = lastMsgId ? [lastMsgId] : [];
            const status = template.status || ((i === message_templates.length - 1) ? 'pending' : 'success');
            const modelId = (status === 'pending' && template.role === 'assistant') ? target_model_id : null;

            newMessages.push({
                role: template.role,
                content: template.content,
                id: currentMsgId,
                evaluationId: null,
                evaluationSessionId: session_id,
                parentMessageIds: parentIds,
                experimental_attachments: [],
                failureReason: null,
                metadata: {},
                modelId: modelId,
                participantPosition: template.participantPosition || "a",
                createdAt: new Date().toISOString(),
                updatedAt: new Date().toISOString(),
                status: status,
            });
            lastMsgId = currentMsgId;
        }

        console.log("[API Bridge] Getting fresh reCAPTCHA token...");
        const recaptchaToken = await getFreshRecaptchaToken();

        if (!recaptchaToken) {
            console.error("[API Bridge] ❌ No reCAPTCHA token available!");
            console.log("[API Bridge] 💡 TIP: Send a message manually first to capture the token parameters");
        } else {
            console.log("[API Bridge] ✅ Got token:", recaptchaToken.substring(0, 20) + "...");
        }

        const formattedHistory = message_templates.map(msg => {
            const roleLabel = msg.role.charAt(0).toUpperCase() + msg.role.slice(1);
            return `${roleLabel}: ${msg.content}`;
        }).join('\n\n');

        const lastUserMsgId = uuidv7();
        const lastAssistantMsgId = uuidv7();

        const body = {
            id: evaluationId,
            mode: "direct",
            modality: "chat",
            modelAId: target_model_id,
            modelAMessageId: lastAssistantMsgId,
            userMessageId: lastUserMsgId,
            userMessage: {
                content: formattedHistory,
                experimental_attachments: []
            },
            recaptchaV3Token: recaptchaToken,
            messages: [],
        };

        console.log("[API Bridge] Sending to LMArena API");
        console.log("[API Bridge] Token length:", recaptchaToken.length);

        window.isApiBridgeRequest = true;
        try {
            const response = await fetch(apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'text/plain;charset=UTF-8',
                    'Accept': '*/*',
                },
                body: JSON.stringify(body),
                credentials: 'include'
            });

            if (!response.ok || !response.body) {
                const errorBody = await response.text();
                throw new Error(`Response error: ${response.status}. ${errorBody}`);
            }

            const reader = response.body.getReader();
            const decoder = new TextDecoder();

            while (true) {
                const { value, done } = await reader.read();
                if (done) {
                    console.log(`[API Bridge] ✅ Request ${requestId.substring(0, 8)} complete`);
                    sendToServer(requestId, "[DONE]");

                    recordRequest();

                    break;
                }
                const chunk = decoder.decode(value);
                sendToServer(requestId, chunk);
            }

        } catch (error) {
            console.error(`[API Bridge] ❌ Error:`, error);

            // Record the attempt (cooldown still applies to prevent spam)
            recordRequest();

            const errMsg = error.message || "";

            // 1. HTTP 429 - Rate Limit
            if (errMsg.includes("429") || errMsg.includes("Too Many Requests")) {
                triggerAutoProtection();
                sendToServer(requestId, { error: `⚠️ AUTOMATIC PROTECTION ENABLED: 429 Detected. System is entering defensive mode. ${errMsg}` });
            }
            // 2. HTTP 500 - Internal Server Error
            else if (errMsg.includes("500") || errMsg.includes("Internal Server Error")) {
                sendToServer(requestId, { error: `⛔ 500 Server Error. Refreshing page...` });
                sendToServer(requestId, "[DONE]");
                setTimeout(() => location.reload(), 1000); // Force refresh
                return;
            }
            // 3. HTTP 403 - reCAPTCHA (Specific validation failure)
            else if (errMsg.includes("403") && errMsg.includes("recaptcha")) {
                sendToServer(requestId, { error: `🚫 reCAPTCHA Failed. Please send a manual message in browser. ${errMsg}` });
                // Do not rollback, this counts as a failed attempt
            }
            // 4. HTTP 422 - Content Filter
            else if (errMsg.includes("422")) {
                rollbackRateLimit();
                sendToServer(requestId, { error: `⚠️ 422 Filter Error: Your message was rejected. TIP: Modify your prompt to prevent getting filtered. ${errMsg}` });
            }
            // 5. Generic / TypeErrors / Input Stream / Network
            else {
                rollbackRateLimit();
                sendToServer(requestId, { error: errMsg });
            }

            sendToServer(requestId, "[DONE]");

        } finally {
            window.isApiBridgeRequest = false;
        }
    }

    // ============================================
    // Utility Functions
    // ============================================
    function sendToServer(requestId, data) {
        if (socket && socket.readyState === WebSocket.OPEN) {
            socket.send(JSON.stringify({ request_id: requestId, data: data }));
        } else {
            console.error("[API Bridge] Cannot send - not connected");
        }
    }

    // ============================================
    // Network Request Interception
    // ============================================
    const originalFetch = window.fetch;
    window.fetch = function(...args) {
        const urlArg = args[0];
        let urlString = '';

        if (urlArg instanceof Request) {
            urlString = urlArg.url;
        } else if (urlArg instanceof URL) {
            urlString = urlArg.href;
        } else if (typeof urlArg === 'string') {
            urlString = urlArg;
        }

        if (urlString && urlString.includes('/create-evaluation') && !window.isApiBridgeRequest) {
            try {
                // Detected manual action
                resetSessionTimer();

                const options = args[1];
                if (options && options.body) {
                    const body = JSON.parse(options.body);
                    if (body.recaptchaV3Token) {
                        window.recaptchaToken = body.recaptchaV3Token;
                        console.log("[API Bridge] 📥 Captured recaptcha token from real request");
                        console.log("[API Bridge]    Token:", body.recaptchaV3Token.substring(0, 30) + "...");
                    }
                }
            } catch (e) {}
        }

        if (urlString) {
            const match = urlString.match(/\/nextjs-api\/stream\/retry-evaluation-session-message\/([a-f0-9-]+)\/messages\/([a-f0-9-]+)/);

            if (match && !window.isApiBridgeRequest && isCaptureModeActive) {
                const sessionId = match[1];
                const messageId = match[2];
                console.log(`[API Bridge] 🎯 Captured IDs in active mode! Sending...`);

                isCaptureModeActive = false;
                if (document.title.startsWith("🎯 ")) {
                    document.title = document.title.substring(2);
                }

                fetch('http://127.0.0.1:5103/update', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ sessionId, messageId })
                })
                .then(response => {
                    if (!response.ok) throw new Error(`Server responded with status: ${response.status}`);
                    console.log(`[API Bridge] ✅ ID update sent successfully. Capture mode auto-disabled.`);
                })
                .catch(err => {
                    console.error('[API Bridge] Error sending ID update:', err.message);
                });
            }
        }

        return originalFetch.apply(this, args);
    };

    // ============================================
    // Page Source Sending (tries both endpoints)
    // ============================================
    async function sendPageSource() {
        const htmlContent = document.documentElement.outerHTML;

        const endpoints = [
            'http://localhost:5102/internal/update_available_models',
            'http://localhost:5102/internal/update_models'
        ];

        let success = false;

        for (const endpoint of endpoints) {
            try {
                const response = await fetch(endpoint, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'text/html; charset=utf-8'
                    },
                    body: htmlContent
                });

                if (response.ok) {
                    console.log(`[API Bridge] ✅ Page source sent successfully to ${endpoint}`);
                    success = true;
                    break;
                }
            } catch (e) {
                console.log(`[API Bridge] ⚠️ Failed to send to ${endpoint}: ${e.message}`);
            }
        }

        if (!success) {
            console.error("[API Bridge] ❌ Failed to send page source to any endpoint");
        }
    }

    // ============================================
    // Initialization
    // ============================================
    console.log("╔══════════════════════════════════════════════════════════════╗");
    console.log("║                 LMArena API Bridge v3.2.6                    ║");
    console.log("╠══════════════════════════════════════════════════════════════╣");
    console.log("║  Type help() in console for commands and settings.           ║");
    console.log("╚══════════════════════════════════════════════════════════════╝");

    connect();

})();